<?php
require_once 'includes/auth.php';
requireLogin();

// Get user stats and info
$pdo = getDBConnection();

// Initialize TaskManager
require_once 'classes/TaskManager.php';
$taskManager = new TaskManager($pdo);

// Get user data and stats
$stmt = $pdo->prepare("SELECT u.*, us.* FROM users u LEFT JOIN user_stats us ON u.id = us.user_id WHERE u.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$userData = $stmt->fetch();

// Ensure we have user stats
if (!$userData) {
    // Initialize user stats if they don't exist
    $stmt = $pdo->prepare("INSERT INTO user_stats (user_id, points, last_updated) VALUES (?, 0, NOW())");
    $stmt->execute([$_SESSION['user_id']]);
    $userData = [
        'points' => 0, 
        'today_completed' => 0,
        'full_name' => 'User',
        'phone' => $_SESSION['phone'] ?? ''
    ];
}

// Get today's tasks for the user (fetch fresh data)
$miniTasks = $taskManager->getDailyTasks($_SESSION['user_id']);

// Get user's task progress (after fetching tasks to ensure fresh data)
$taskProgress = $taskManager->getUserTaskProgress($_SESSION['user_id']);
$completedTasks = $taskProgress['completed'];
$totalTasks = $taskProgress['total'];
$currentTaskIndex = $taskProgress['next_index'];

// Get the current task (next uncompleted task)
$currentTask = null;
$taskIndex = 1;
$allTasksCompleted = false;

foreach ($miniTasks as $task) {
    if ($task['status'] === 'assigned') {
        $currentTask = $task;
        $currentTaskIndex = $taskIndex; // Use the actual position in the list
        break;
    }
    $taskIndex++;
}

// Check if all tasks are completed
if (!$currentTask && !empty($miniTasks) && $completedTasks >= $totalTasks) {
    $allTasksCompleted = true;
    $currentTask = end($miniTasks); // Keep last task for display
    $currentTaskIndex = $totalTasks;
    
    // Note: Streak is updated in save_recording.php when the last task is completed
    // This prevents duplicate increments on page reloads
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>Zeba Voice Proj 1</title>
<style>
:root{
  --bg:#000000; --surface:#1a1a1a; --muted:#2a2a2a; --accent-orange:#FFC929; --accent-blue:#1ea7ff; --user-bg:#222222;
  --accent-yellow:#ffd33d; --approved:#27ae60; --rejected:#e74c3c; --glass: rgba(255,255,255,0.03);
  --success: #10b981;
  --danger: #ef4444;
  --radius: 12px;
  --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
  --purple: #a78bfa;
  --pink: #f472b6;
}

html,body{
  height:100%;
  margin:0;
  background: radial-gradient(ellipse at top, #1a0a00 0%, #000000 50%, #000000 100%);
  color:#fff;
  font-family:Inter, "Segoe UI", Roboto, system-ui, -apple-system, "Helvetica Neue", Arial;
  -webkit-font-smoothing:antialiased;
  -moz-osx-font-smoothing:grayscale;
  position: relative;
  overflow: hidden;
}

/* Animated particles */
body::before {
  content: '';
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-image: 
    radial-gradient(circle at 20% 30%, rgba(255, 201, 41, 0.08) 0%, transparent 40%),
    radial-gradient(circle at 80% 70%, rgba(255, 211, 61, 0.05) 0%, transparent 40%),
    radial-gradient(circle at 50% 50%, rgba(255, 201, 41, 0.03) 0%, transparent 50%);
  animation: particleGlow 15s ease-in-out infinite;
  pointer-events: none;
  z-index: 0;
}

@keyframes particleGlow {
  0%, 100% { opacity: 1; transform: scale(1); }
  50% { opacity: 0.7; transform: scale(1.1); }
}

/* Header + streak bar */
header.appbar{
  padding:16px 20px;
  background: linear-gradient(135deg, rgba(26, 10, 0, 0.95) 0%, rgba(0, 0, 0, 0.95) 100%);
  backdrop-filter: blur(20px);
  -webkit-backdrop-filter: blur(20px);
  display:flex;
  flex-direction:column;
  gap:12px;
  border-bottom:2px solid rgba(255, 201, 41, 0.4);
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.6), 0 2px 4px rgba(255, 201, 41, 0.2);
  position: relative;
  z-index: 10;
  animation: slideDown 0.6s cubic-bezier(0.68, -0.55, 0.265, 1.55);
}

@keyframes slideDown {
  from { transform: translateY(-100%); opacity: 0; }
  to { transform: translateY(0); opacity: 1; }
}
.app-title{
  display:flex;
  align-items:center;
  gap:10px;
  font-weight:700;
  font-size:18px;
}
.app-title .logo{
  width:44px;height:44px;border-radius:12px;
  background: linear-gradient(135deg, var(--accent-orange) 0%, #FFB800 100%);
  background-size: 200% 200%;
  animation: logoShine 4s ease-in-out infinite;
  display:inline-flex;align-items:center;justify-content:center;font-weight:700;color:#000;
  box-shadow: 0 4px 16px rgba(255, 201, 41, 0.6), inset 0 1px 0 rgba(255,255,255,0.3);
  transition: transform 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
}

.app-title .logo:hover {
  transform: rotate(8deg) scale(1.1);
  box-shadow: 0 6px 24px rgba(255, 201, 41, 0.8), inset 0 1px 0 rgba(255,255,255,0.4);
}

@keyframes logoShine {
  0%, 100% { background-position: 0% 50%; }
  50% { background-position: 100% 50%; }
}

/* Streak row - fixed horizontal - ALWAYS one line */
.streak-row{
  display:flex;
  gap:10px;
  justify-content:space-between;
  width:100%;
  flex-wrap: nowrap;
}
.streak-item{
  flex:1 1 0;
  min-width:0;
  background: linear-gradient(135deg, rgba(26, 26, 26, 0.9) 0%, rgba(20, 10, 5, 0.9) 100%);
  backdrop-filter: blur(10px);
  padding:12px 16px;
  border-radius:12px;
  font-size:13px;
  color:#ffffff;
  border:1px solid rgba(255, 201, 41, 0.4);
  overflow:hidden;
  text-overflow:ellipsis;
  white-space:nowrap;
  transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
  position: relative;
  animation: fadeSlideUp 0.6s ease-out backwards;
  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.4), inset 0 1px 0 rgba(255, 201, 41, 0.1);
}

.streak-item:nth-child(1) { animation-delay: 0.1s; }
.streak-item:nth-child(2) { animation-delay: 0.2s; }
.streak-item:nth-child(3) { animation-delay: 0.3s; }

.streak-item:hover {
  transform: translateY(-3px);
  border-color: var(--accent-orange);
  box-shadow: 0 6px 20px rgba(255, 201, 41, 0.4), inset 0 1px 0 rgba(255, 201, 41, 0.2);
  background: linear-gradient(135deg, rgba(30, 30, 30, 0.95) 0%, rgba(26, 13, 6, 0.95) 100%);
}

@keyframes fadeSlideUp {
  from { opacity: 0; transform: translateY(20px); }
  to { opacity: 1; transform: translateY(0); }
}
.streak-amount{
  font-weight:800;
  background: linear-gradient(135deg, #FFD54F 0%, var(--accent-orange) 50%, var(--accent-yellow) 100%);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  background-clip: text;
  margin-left:4px;
  font-size: 1.15em;
  text-shadow: 0 0 20px rgba(255, 201, 41, 0.5);
  animation: glow 2s ease-in-out infinite;
}

@keyframes glow {
  0%, 100% { filter: brightness(1); }
  50% { filter: brightness(1.2); }
}
.streak-small{font-size:12px;color:var(--accent-orange);margin-left:6px;opacity:0.8;}

/* Celebration Screen Styles */
.celebration-container {
  position: relative;
  width: 100%;
  display: flex;
  align-items: center;
  justify-content: center;
  padding: 8px;
  animation: fadeIn 0.6s ease-out;
  overflow: hidden;
}

@keyframes fadeIn {
  from { opacity: 0; transform: scale(0.9); }
  to { opacity: 1; transform: scale(1); }
}

.celebration-card {
  background: linear-gradient(135deg, rgba(255, 201, 41, 0.15) 0%, rgba(255, 184, 0, 0.15) 100%);
  backdrop-filter: blur(20px);
  border-radius: 16px;
  padding: 16px 14px;
  max-width: 95%;
  width: 100%;
  border: 2px solid rgba(255, 201, 41, 0.5);
  box-shadow: 0 20px 60px rgba(255, 201, 41, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.1);
  text-align: center;
  position: relative;
  z-index: 2;
  animation: cardBounce 0.8s cubic-bezier(0.68, -0.55, 0.265, 1.55);
}

@keyframes cardBounce {
  0% { transform: scale(0.5) rotate(-5deg); opacity: 0; }
  50% { transform: scale(1.05) rotate(2deg); }
  100% { transform: scale(1) rotate(0deg); opacity: 1; }
}

.celebration-icon {
  font-size: 36px;
  margin-bottom: 8px;
  animation: iconSpin 1s ease-out, iconBounce 2s ease-in-out 1s infinite;
  display: inline-block;
}

@keyframes iconSpin {
  0% { transform: rotate(0deg) scale(0); }
  50% { transform: rotate(180deg) scale(1.2); }
  100% { transform: rotate(360deg) scale(1); }
}

@keyframes iconBounce {
  0%, 100% { transform: translateY(0) rotate(0deg); }
  50% { transform: translateY(-10px) rotate(10deg); }
}

.celebration-title {
  font-size: 22px;
  font-weight: 800;
  background: linear-gradient(135deg, var(--accent-orange) 0%, var(--accent-yellow) 100%);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  background-clip: text;
  margin-bottom: 8px;
  animation: titleGlow 2s ease-in-out infinite;
}

@keyframes titleGlow {
  0%, 100% { filter: drop-shadow(0 0 10px rgba(255, 201, 41, 0.5)); }
  50% { filter: drop-shadow(0 0 20px rgba(255, 201, 41, 0.8)); }
}

.celebration-message {
  font-size: 13px;
  color: #e0e0e0;
  margin-bottom: 12px;
  line-height: 1.4;
}

.highlight-number {
  font-size: 18px;
  font-weight: 700;
  color: var(--accent-orange);
  text-shadow: 0 0 10px rgba(255, 201, 41, 0.5);
}

.celebration-stats {
  display: flex;
  gap: 8px;
  justify-content: center;
  margin-bottom: 12px;
  flex-wrap: wrap;
}

.stat-item {
  background: rgba(0, 0, 0, 0.4);
  backdrop-filter: blur(10px);
  border-radius: 10px;
  padding: 10px 12px;
  min-width: 85px;
  flex: 1;
  max-width: 110px;
  border: 1px solid rgba(255, 201, 41, 0.3);
  transition: all 0.3s ease;
  animation: statPop 0.6s ease-out backwards;
}

.stat-item:nth-child(1) { animation-delay: 0.2s; }
.stat-item:nth-child(2) { animation-delay: 0.3s; }
.stat-item:nth-child(3) { animation-delay: 0.4s; }

@keyframes statPop {
  0% { transform: scale(0) rotate(-180deg); opacity: 0; }
  70% { transform: scale(1.1) rotate(10deg); }
  100% { transform: scale(1) rotate(0deg); opacity: 1; }
}

.stat-item:hover {
  transform: translateY(-5px) scale(1.05);
  border-color: var(--accent-orange);
  box-shadow: 0 10px 30px rgba(255, 201, 41, 0.4);
}

.stat-icon {
  font-size: 20px;
  margin-bottom: 4px;
}

.stat-value {
  font-size: 18px;
  font-weight: 700;
  color: var(--accent-orange);
  margin-bottom: 2px;
  text-shadow: 0 2px 10px rgba(255, 201, 41, 0.5);
}

.stat-label {
  font-size: 10px;
  color: #b0b0b0;
  text-transform: uppercase;
  letter-spacing: 0.3px;
}

.celebration-footer {
  background: linear-gradient(135deg, rgba(30, 15, 5, 0.6) 0%, rgba(20, 20, 20, 0.6) 100%);
  backdrop-filter: blur(10px);
  border-radius: 8px;
  padding: 10px;
  border-left: 3px solid var(--accent-orange);
}

.next-task-info {
  font-size: 12px;
  color: #e0e0e0;
  line-height: 1.3;
}

/* Confetti Animation */
.confetti {
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  pointer-events: none;
  z-index: 1;
  overflow: hidden;
}

.confetti-piece {
  position: absolute;
  width: 10px;
  height: 10px;
  background: var(--accent-orange);
  top: -10px;
  opacity: 0;
  animation: confettiFall 3s ease-out forwards;
}

@keyframes confettiFall {
  0% { 
    top: -10px; 
    opacity: 1;
    transform: translateX(0) rotateZ(0deg);
  }
  100% { 
    top: 100vh; 
    opacity: 0;
    transform: translateX(var(--confetti-x)) rotateZ(720deg);
  }
}

/* Mobile responsive */
@media (max-width: 480px) {
  .celebration-container {
    padding: 5px;
  }
  
  .celebration-card {
    padding: 14px 12px;
    border-radius: 14px;
  }
  
  .celebration-icon {
    font-size: 32px;
    margin-bottom: 6px;
  }
  
  .celebration-title {
    font-size: 20px;
    margin-bottom: 6px;
  }
  
  .celebration-message {
    font-size: 12px;
    margin-bottom: 10px;
  }
  
  .highlight-number {
    font-size: 16px;
  }
  
  .celebration-stats {
    gap: 6px;
    margin-bottom: 10px;
  }
  
  .stat-item {
    min-width: 75px;
    padding: 8px 10px;
    border-radius: 8px;
  }
  
  .stat-icon {
    font-size: 18px;
    margin-bottom: 3px;
  }
  
  .stat-value {
    font-size: 16px;
  }
  
  .stat-label {
    font-size: 9px;
  }
  
  .celebration-footer {
    padding: 8px;
  }
  
  .next-task-info {
    font-size: 11px;
  }
}

/* Chat area - Fixed height to ensure button is visible */
.container{
  display:flex;
  flex-direction:column;
  height: calc(100vh - 130px);
  position: relative;
  z-index: 1;
}
.chat-wrap{
  flex:1;
  overflow-y:auto;
  overflow-x:hidden;
  padding:18px;
  display:flex;
  flex-direction:column;
  gap:14px;
  scroll-behavior:smooth;
}

.bubble{
  max-width:78%;
  padding:16px 18px;
  border-radius:16px;
  font-size:15px;
  line-height:1.5;
  box-shadow: 0 8px 32px rgba(0,0,0,0.5);
  word-break:break-word;
  transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
  position: relative;
  animation: bubblePopIn 0.4s cubic-bezier(0.4, 0, 0.2, 1);
  z-index: 1;
}

.bubble:hover {
  transform: translateY(-3px) scale(1.01);
  box-shadow: 0 12px 48px rgba(0,0,0,0.6);
}

@keyframes bubblePopIn {
  0% { transform: scale(0.95); opacity: 0; }
  100% { transform: scale(1); opacity: 1; }
}

/* system welcome (orange strip on left) */
.bubble.welcome{
  align-self:flex-start;
  background: linear-gradient(135deg, rgba(26, 10, 0, 0.9) 0%, rgba(20, 20, 20, 0.9) 100%);
  backdrop-filter: blur(15px);
  color:#ffffff;
  font-weight:500;
  border-radius:16px;
  padding:16px 18px;
  margin-bottom: 8px;
  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-left: 5px solid var(--accent-orange);
}

.bubble.instruction{
  align-self:flex-start;
  background: linear-gradient(135deg, rgba(30, 167, 255, 0.05) 0%, rgba(20, 20, 20, 0.9) 100%);
  backdrop-filter: blur(20px);
  color:#ffffff;
  border: 1px solid rgba(30, 167, 255, 0.12);
  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1), 0 0 0 1px rgba(30, 167, 255, 0.05);
  font-size: 16px;
  font-weight: 500;
  line-height: 1.6;
  letter-spacing: 0.3px;
  position: relative;
  padding: 18px 22px;
  margin-bottom: 10px;
  border-radius: 12px;
  border-left: 3px solid rgba(30, 167, 255, 0.4);
}

@keyframes instructionPulse {
  0%, 100% { 
    border-color: rgba(30, 167, 255, 0.5);
    box-shadow: 0 12px 48px rgba(30, 167, 255, 0.25), inset 0 1px 0 rgba(30, 167, 255, 0.2), -4px 0 20px rgba(30, 167, 255, 0.4);
  }
  50% { 
    border-color: rgba(30, 167, 255, 0.7);
    box-shadow: 0 12px 48px rgba(30, 167, 255, 0.35), inset 0 1px 0 rgba(30, 167, 255, 0.3), -4px 0 25px rgba(30, 167, 255, 0.5);
  }
}

.bubble.instruction::before {
  content: '📌';
  position: absolute;
  left: -14px;
  top: 10px;
  font-size: 16px;
  background: var(--accent-blue);
  width: 26px;
  height: 26px;
  border-radius: 8px;
  display: flex;
  align-items: center;
  justify-content: center;
  box-shadow: 0 4px 12px rgba(30, 167, 255, 0.3), 0 0 15px rgba(30, 167, 255, 0.2);
  animation: iconBob 2.5s ease-in-out infinite;
  z-index: 10;
  transform: rotate(-5deg);
}

@keyframes iconBob {
  0%, 100% { transform: translateY(0) rotate(0deg); }
  50% { transform: translateY(-3px) rotate(-5deg); }
}

.bubble.instruction:hover {
  border-color: rgba(30, 167, 255, 0.2);
  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15), 0 0 0 1px rgba(30, 167, 255, 0.1);
  background: linear-gradient(135deg, rgba(30, 167, 255, 0.08) 0%, rgba(20, 20, 20, 0.92) 100%);
}

/* Example Audio Bubble */
.bubble.example-audio {
  align-self: flex-start;
  background: linear-gradient(135deg, rgba(26, 10, 0, 0.85) 0%, rgba(20, 20, 20, 0.85) 100%);
  backdrop-filter: blur(15px);
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 16px;
  padding: 18px;
  margin-top: 12px;
  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
  max-width: 78%;
  box-sizing: border-box;
  transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
  position: relative;
  overflow: visible;
  margin-bottom: 16px;
}

/* Removed glow effect from example audio */

@keyframes ripple {
  0%, 100% { transform: translate(0, 0); }
  50% { transform: translate(10%, 10%); }
}

.bubble.example-audio:hover {
  border-color: rgba(255, 201, 41, 0.3);
  box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
  transform: translateY(-1px);
}

.example-header {
  display: flex;
  align-items: center;
  margin-bottom: 14px;
  color: var(--accent-orange);
  font-weight: 600;
  font-size: 16px;
  text-shadow: 0 2px 8px rgba(255, 201, 41, 0.4);
  position: relative;
  z-index: 1;
  .task-counter {
    font-size: 0.9em;
    color: #aaa;
    margin: 10px 0 5px;
    font-weight: 500;
  }
  .task-counter .completed {
    color: var(--accent-orange);
    font-weight: 700;
    font-size: 1.1em;
  }
  .task-counter .total {
    color: #777;
    font-weight: 500;
  }
  .task-progress {
    font-size: 0.85em;
    color: var(--accent-blue);
    margin-bottom: 10px;
    font-weight: 500;
  }
  .task-progress:before {
    content: '•';
    margin-right: 5px;
    color: var(--accent-blue);
  }
}

.example-icon {
  margin-right: 10px;
  font-size: 22px;
  animation: iconBounce 2.5s ease-in-out infinite;
  display: inline-block;
}

@keyframes iconBounce {
  0%, 100% { transform: translateY(0) rotate(0deg); }
  25% { transform: translateY(-4px) rotate(-5deg); }
  75% { transform: translateY(-2px) rotate(5deg); }
}

.audio-player-wrapper {
  background: rgba(30, 30, 30, 0.8);
  backdrop-filter: blur(10px);
  border-radius: 14px;
  padding: 14px;
  margin: 10px 0;
  border: 1px solid rgba(255, 255, 255, 0.15);
  transition: all 0.3s ease;
  box-shadow: inset 0 2px 8px rgba(0, 0, 0, 0.3);
  position: relative;
  z-index: 1;
}

.audio-player-wrapper:hover {
  background: rgba(35, 35, 35, 0.9);
  border-color: rgba(255, 201, 41, 0.5);
  box-shadow: inset 0 2px 8px rgba(0, 0, 0, 0.3), 0 0 15px rgba(255, 201, 41, 0.2);
}

.custom-audio-player {
  width: 100%;
  height: 40px;
}

/* Style the audio player controls */
custom-audio-player::-webkit-media-controls-panel {
  background: transparent;
}

custom-audio-player::-webkit-media-controls-play-button,
custom-audio-player::-webkit-media-controls-current-time-display,
custom-audio-player::-webkit-media-controls-time-remaining-display {
  color: var(--accent-blue);
}

.example-footer {
  margin-top: 12px;
  font-size: 13px;
  color: #e0e0e0;
  display: flex;
  align-items: flex-start;
  gap: 8px;
  padding: 12px;
  background: linear-gradient(135deg, rgba(30, 15, 5, 0.6) 0%, rgba(20, 20, 20, 0.6) 100%);
  backdrop-filter: blur(8px);
  border-radius: 10px;
  border-left: 3px solid var(--accent-orange);
  box-shadow: inset 0 1px 3px rgba(0, 0, 0, 0.3);
  position: relative;
  z-index: 1;
}

.example-tip {
  font-style: italic;
  opacity: 0.95;
  line-height: 1.5;
}

/* Dark theme for audio player */
audio {
  filter: invert(1) hue-rotate(180deg) brightness(1.2) contrast(0.9);
}

.bubble.system{
  align-self:flex-start;
  background: linear-gradient(135deg, rgba(26, 26, 26, 0.95) 0%, rgba(20, 20, 20, 0.95) 100%);
  backdrop-filter: blur(15px);
  color:#ffffff;
  border: 1px solid rgba(255, 255, 255, 0.3);
  box-shadow: 0 8px 32px rgba(0, 0, 0, 0.4), inset 0 1px 0 rgba(255, 255, 255, 0.1);
  margin-top: 12px;
}

.bubble.user{
  align-self:flex-end;
  background: linear-gradient(135deg, rgba(42, 42, 42, 0.95) 0%, rgba(30, 15, 5, 0.95) 100%);
  backdrop-filter: blur(15px);
  color:#fff;
  border:1px solid rgba(255, 201, 41, 0.5);
  position: relative;
  margin-left:auto;
  margin-top: 12px;
  max-width:80%;
  border-radius:16px 16px 4px 16px;
  padding:14px 18px;
  box-shadow: 0 8px 32px rgba(255, 201, 41, 0.2), inset 0 1px 0 rgba(255, 201, 41, 0.15);
  clear: both;
}

.bubble.user:hover {
  border-color: var(--accent-orange);
  box-shadow: 0 12px 48px rgba(255, 201, 41, 0.3), inset 0 1px 0 rgba(255, 201, 41, 0.2);
  transform: translateY(-2px);
}

.bubble.user::after {
  content: '';
  position: absolute;
  top: 0;
  right: -4px;
  height: 100%;
  width: 4px;
  background: linear-gradient(180deg, var(--accent-orange) 0%, var(--accent-yellow) 100%);
  border-radius: 0 14px 14px 0;
  box-shadow: 0 0 12px rgba(255, 201, 41, 0.6);
}

.meta{
  display:block;
  margin-top:8px;
  font-size:12px;
  color:#b8c1d9;
}

.audio-preview{
  margin-top:6px;
  width:100%;
  border-radius:8px;
  background:rgba(255,255,255,0.02);
  padding:6px;
}

.status-pending{ color:#f6c85f; font-weight:600; }
.status-approved{ color:var(--approved); font-weight:700; }
.status-rejected{ color:var(--rejected); font-weight:700; }

/* record area bottom */
.controls{
  padding:18px;
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  gap: 10px;
  border-top: 1px solid rgba(255,255,255,0.03);
  text-align: center;
  min-height: 120px;
}

.review-container {
  display: none;
  width: 100%;
  max-width: 500px;
  background: linear-gradient(135deg, rgba(30, 30, 30, 0.95) 0%, rgba(20, 20, 20, 0.95) 100%);
  backdrop-filter: blur(20px);
  border-radius: 16px;
  padding: 20px;
  border: 2px solid rgba(255, 201, 41, 0.4);
  box-shadow: 0 8px 32px rgba(255, 201, 41, 0.3);
  animation: slideUp 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

@keyframes slideUp {
  from { transform: translateY(20px); opacity: 0; }
  to { transform: translateY(0); opacity: 1; }
}

.review-container.active {
  display: flex;
  flex-direction: column;
  gap: 16px;
}

.review-header {
  color: var(--accent-orange);
  font-weight: 600;
  font-size: 16px;
  text-align: center;
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 8px;
}

.review-audio-player {
  background: rgba(0, 0, 0, 0.4);
  border-radius: 12px;
  padding: 12px;
  border: 1px solid rgba(255, 255, 255, 0.1);
}

.review-audio-player audio {
  width: 100%;
  height: 40px;
}

.review-buttons {
  display: flex;
  gap: 12px;
  justify-content: center;
}

.review-buttons .btn {
  flex: 1;
  max-width: 180px;
  padding: 12px 24px;
  border-radius: 12px;
  border: none;
  font-weight: 600;
  font-size: 15px;
  cursor: pointer;
  transition: all 0.3s ease;
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 8px;
}

.review-buttons .btn-submit {
  background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
  color: white;
  box-shadow: 0 4px 16px rgba(16, 185, 129, 0.4);
}

.review-buttons .btn-submit:hover {
  transform: translateY(-2px) scale(1.05);
  box-shadow: 0 6px 24px rgba(16, 185, 129, 0.6);
}

.review-buttons .btn-delete {
  background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%);
  color: white;
  box-shadow: 0 4px 16px rgba(239, 68, 68, 0.4);
}

.review-buttons .btn-delete:hover {
  transform: translateY(-2px) scale(1.05);
  box-shadow: 0 6px 24px rgba(239, 68, 68, 0.6);
}

.recording-controls {
  display: flex;
  flex-direction: column;
  align-items: center;
  gap: 10px;
}

.mic-btn {
  width: 100px;
  height: 100px;
  border-radius: 50%;
  border: none;
  background: linear-gradient(135deg, #FEF08A 0%, #F59E0B 100%);
  color: #1f2937;
  font-size: 36px;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  position: relative;
  overflow: hidden;
  box-shadow: 0 8px 32px rgba(245, 158, 11, 0.2), 
              0 0 0 1px rgba(255, 255, 255, 0.1),
              inset 0 0 0 1px rgba(255, 255, 255, 0.1);
  transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
  z-index: 1;
  backdrop-filter: blur(4px);
  -webkit-backdrop-filter: blur(4px);
}

.mic-btn .pulse-ring {
  content: '';
  position: absolute;
  width: 100%;
  height: 100%;
  background: rgba(255, 255, 255, 0.1);
  border-radius: 50%;
  opacity: 0;
  transition: all 0.6s ease-out;
  z-index: -1;
}

.mic-btn:hover .pulse-ring {
  opacity: 0.6;
  transform: scale(1.2);
}

.mic-btn.recording .pulse-ring {
  animation: sonar 2s infinite;
}

@keyframes sonar {
  0% {
    transform: scale(1);
    opacity: 0.6;
  }
  100% {
    transform: scale(1.5);
    opacity: 0;
  }
}

.mic-btn:hover::before {
  transform: translate(-50%, -50%) scale(1);
  opacity: 0.3;
}

.mic-btn.recording {
  background: linear-gradient(135deg, #F87171 0%, #DC2626 100%);
  animation: pulse 1.5s infinite;
  box-shadow: 0 0 0 0 rgba(220, 38, 38, 0.7);
}

@keyframes pulse {
  0% {
    transform: scale(0.95);
    box-shadow: 0 0 0 0 rgba(220, 38, 38, 0.7);
  }
  70% {
    transform: scale(1);
    box-shadow: 0 0 0 15px rgba(220, 38, 38, 0);
  }
  100% {
    transform: scale(0.95);
    box-shadow: 0 0 0 0 rgba(220, 38, 38, 0);
  }
}

.timer{
  background: linear-gradient(135deg, #FFD54F 0%, var(--accent-orange) 50%, var(--accent-yellow) 100%);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  background-clip: text;
  font-weight: 800;
  font-size: 22px;
  margin-bottom: 10px;
  text-align: center;
  width: 100%;
  letter-spacing: 3px;
  font-family: 'Courier New', monospace;
  text-shadow: 0 0 30px rgba(255, 201, 41, 0.8);
  animation: timerPulse 1s ease-in-out infinite;
}

@keyframes timerPulse {
  0%, 100% { opacity: 1; }
  50% { opacity: 0.85; }
}

/* Subtle scrollbar */
.chat-wrap::-webkit-scrollbar {
  width: 6px;
}

.chat-wrap::-webkit-scrollbar-track {
  background: rgba(40, 40, 40, 0.3);
  border-radius: 3px;
}

.chat-wrap::-webkit-scrollbar-thumb {
  background: rgba(120, 120, 120, 0.5);
  border-radius: 3px;
  transition: all 0.3s ease;
  border: 1px solid rgba(255, 255, 255, 0.1);
}

.chat-wrap::-webkit-scrollbar-thumb:hover {
  background: var(--accent-yellow);
  opacity: 0.7;
}

/* Loading spinner for audio */
.audio-loading {
  display: inline-block;
  width: 20px;
  height: 20px;
  border: 3px solid rgba(168, 139, 250, 0.3);
  border-radius: 50%;
  border-top-color: var(--purple);
  animation: spinner 0.8s linear infinite;
}

@keyframes spinner {
  to { transform: rotate(360deg); }
}

/* Progress indicator */
.progress-ring {
  transform: rotate(-90deg);
  transform-origin: center;
}

.progress-ring circle {
  transition: stroke-dashoffset 0.5s ease;
}

/* small mobile tweaks */
@media (max-width:480px){
  .streak-row{ gap:6px; flex-wrap: nowrap; }
  .streak-item { 
    min-width:0;
    padding: 10px 12px;
    font-size: 12px;
  }
  .streak-amount { font-size: 1em; }
  .chat-wrap{ padding:14px; }
  .bubble{ font-size:14px; max-width: 85%; padding: 14px 16px; }
  .mic-btn{ width:75px;height:75px;font-size:30px; }
  header.appbar { padding: 12px 16px; }
  .app-title { font-size: 16px; }
  .app-title .logo { width: 38px; height: 38px; }
  .nav-links { gap: 8px; }
  .nav-link { padding: 8px 14px; font-size: 13px; }
  .timer { font-size: 18px; letter-spacing: 2px; }
}

/* Modern navigation links */
.nav-links {
  margin-left: auto;
  display: flex;
  gap: 12px;
  align-items: center;
}

.nav-link {
  padding: 10px 18px;
  border-radius: 12px;
  text-decoration: none;
  font-weight: 600;
  font-size: 14px;
  transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
  position: relative;
  overflow: hidden;
}

.nav-link::before {
  content: '';
  position: absolute;
  top: 0;
  left: -100%;
  width: 100%;
  height: 100%;
  background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
  transition: left 0.5s ease;
}

.nav-link:hover::before {
  left: 100%;
}

.nav-link.logout {
  color: var(--accent-orange);
  background: linear-gradient(135deg, rgba(255, 201, 41, 0.15) 0%, rgba(255, 184, 0, 0.15) 100%);
  backdrop-filter: blur(10px);
  border: 1px solid rgba(255, 201, 41, 0.4);
  box-shadow: 0 2px 8px rgba(255, 201, 41, 0.2), inset 0 1px 0 rgba(255, 201, 41, 0.2);
}

.nav-link.logout:hover {
  background: linear-gradient(135deg, rgba(255, 201, 41, 0.25) 0%, rgba(255, 184, 0, 0.25) 100%);
  border-color: var(--accent-orange);
  transform: translateY(-2px) scale(1.05);
  box-shadow: 0 6px 20px rgba(255, 201, 41, 0.4), inset 0 1px 0 rgba(255, 201, 41, 0.3);
}
</style>
</head>
<body>
<header class="appbar">
  <div class="app-title">
    <div class="logo">ZV</div>
    <span>Zeba Voice Proj 1</span>
    <div class="nav-links">
      <a href="logout.php" class="nav-link logout">🚪 Logout</a>
    </div>
  </div>
  <div class="streak-row" aria-hidden="true">
    <div class="streak-item">
      <span>Daily Task: <span class="streak-amount"><?php echo $completedTasks; ?>/<?php echo $totalTasks; ?></span></span>
    </div>
    <div class="streak-item">
      <span>Score: <span class="streak-amount"><?php echo $userData['points'] ?? 0; ?></span><span class="streak-small">pts</span></span>
    </div>
    <div class="streak-item">
      <span>Streaks: <span class="streak-amount"><?php echo $userData['consecutive_days'] ?? 0; ?></span><span class="streak-small">days</span></span>
    </div>
  </div>
</header>

<div class="container">
  <div class="chat-wrap" id="chat">
    <?php if (!$allTasksCompleted): ?>
    <!-- Welcome (orange left strip) -->
    <div class="bubble welcome" id="welcomeBubble">
      👋 Welcome back, <strong style="color: var(--accent-yellow);"><?php echo htmlspecialchars($userData['full_name'] ?? 'User'); ?></strong>! Your recordings power Zeba's voice intelligence, helping it understand Ghanaian languages naturally. 🙏 We truly appreciate your contribution.
      <div class="meta">💡 <strong>Tip:</strong> Just speak naturally we'll handle the rest.</div>
    </div>
    <?php endif; ?>

    <?php if ($allTasksCompleted): ?>
    <!-- Epic Celebration Screen -->
    <div class="celebration-container" id="celebrationContainer">
      <div class="celebration-card">
        <div class="celebration-icon">🎉</div>
        <h2 class="celebration-title">Congratulations!</h2>
        <p class="celebration-message">
          You've crushed it today! All <span class="highlight-number"><?php echo $totalTasks; ?></span> tasks completed!
        </p>
        <div class="celebration-stats">
          <div class="stat-item">
            <div class="stat-icon">✅</div>
            <div class="stat-value"><?php echo $completedTasks; ?>/<?php echo $totalTasks; ?></div>
            <div class="stat-label">Tasks Done</div>
          </div>
          <div class="stat-item">
            <div class="stat-icon">🔥</div>
            <div class="stat-value"><?php echo $userData['consecutive_days'] ?? 0; ?></div>
            <div class="stat-label">Day Streak</div>
          </div>
          <div class="stat-item">
            <div class="stat-icon">⭐</div>
            <div class="stat-value"><?php echo $userData['points'] ?? 0; ?></div>
            <div class="stat-label">Points</div>
          </div>
        </div>
        <div class="celebration-footer">
          <div class="next-task-info">
            📅 Come back tomorrow for more tasks and keep your streak going!
          </div>
        </div>
      </div>
      <!-- Confetti Animation -->
      <div class="confetti" id="confetti"></div>
    </div>
    <?php elseif ($currentTask): ?>
    <!-- Task Instruction -->
    <div class="bubble instruction">
      <div style="font-size: 0.95em; opacity: 0.9; margin-bottom: 10px; line-height: 1.5;">
        Listen to the audio and record the phrases you hear. Try to match the tone and pacing of audio.
      </div>
      <div style="font-size: 1.05em; margin-bottom: 8px; line-height: 1.5;">
        <span style="color: #ffffff; font-weight: 700;">Say: </span>
        <span style="color: #2cbf78ff; font-weight: 700;"><?php echo htmlspecialchars($currentTask['instruction_text']); ?></span>
      </div>
      <div class="meta" style="color: rgba(30, 167, 255, 0.8); font-weight: 600; margin-top: 8px;">📍 Task <?php echo $currentTaskIndex . ' of ' . $totalTasks; ?> • ⏱️ Keep recordings under ~5s</div>
    </div>

    <?php if (!empty($currentTask['sample_audio_url'])): ?>
    <!-- Example audio below instruction -->
    <div class="bubble example-audio">
      <div class="example-header">
        <div class="example-icon">🎧</div>
        <div class="example-title">Example Recording</div>
      </div>
      <div class="audio-player-wrapper">
        <audio class="custom-audio-player" controls>
          <source src="<?php 
            $audioUrl = $currentTask['sample_audio_url'] ?? '';
            
            if ($audioUrl) {
                // Log the original URL for debugging
                error_log("Original audio URL from DB: " . $audioUrl);
                
                // Remove any leading slashes or uploads/ prefix to normalize
                $audioUrl = ltrim($audioUrl, '/');
                $audioUrl = preg_replace('|^uploads/|', '', $audioUrl);
                
                // Define the correct path to the uploads directory
                $uploadDir = '/uploads/';
                $audioFile = $audioUrl;
                
                // Build the full server path to check if file exists
                $fullPath = $_SERVER['DOCUMENT_ROOT'] . $uploadDir . $audioFile;
                
                // Log the paths for debugging
                error_log("Looking for audio file at: " . $fullPath);
                
                // Check if file exists and is readable
                if (file_exists($fullPath) && is_readable($fullPath)) {
                    $webPath = $uploadDir . $audioFile;
                    error_log("Audio file found, serving from: " . $webPath);
                    echo htmlspecialchars($webPath);
                } else {
                    // Try alternative locations if the file isn't found
                    $alternativePaths = [
                        '/uploads/' . basename($audioUrl),
                        '/vonaida/uploads/' . $audioFile,
                        '/vonaida/uploads/' . basename($audioUrl),
                        $audioUrl,
                        '/vonaida/' . $audioUrl
                    ];
                    
                    $found = false;
                    foreach ($alternativePaths as $altPath) {
                        $altFullPath = $_SERVER['DOCUMENT_ROOT'] . $altPath;
                        if (file_exists($altFullPath) && is_readable($altFullPath)) {
                            error_log("Found audio at alternative path: " . $altPath);
                            echo htmlspecialchars($altPath);
                            $found = true;
                            break;
                        }
                    }
                    
                    if (!$found) {
                        error_log("Audio file not found at any location. Tried:" . print_r($alternativePaths, true));
                        echo '#'; // Fallback to prevent broken audio element
                    }
                }
            } else {
                error_log("No audio URL provided for task ID: " . ($currentTask['id'] ?? 'unknown'));
                echo '#'; // Fallback to prevent broken audio element
            }
          ?>" type="audio/wav">
          Your browser does not support the audio element.
        </audio>
      </div>
      <?php if (!empty($currentTask['tips_text'])): ?>
      <div class="example-footer">
        <span class="example-tip">💡 Listen, then record your version.</span>
      </div>
      <?php endif; ?>
    </div>
    <?php endif; ?>
    <?php else: ?>
    <!-- Fallback if no tasks available -->
    <div class="bubble instruction">
      No tasks available at the moment. Please check back later.
      <div class="meta">Task 0 of 0</div>
    </div>
    <?php endif; ?>
  </div>

  <div class="controls">
    <div class="recording-controls" id="recordingControls">
      <div class="timer" id="timerDisplay">00:00</div>
      <button id="micBtn" class="mic-btn" aria-pressed="false" <?php echo $allTasksCompleted ? 'disabled style="opacity: 0.3; cursor: not-allowed;"' : ''; ?>>
        <span class="mic-icon">
          <svg width="36" height="36" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M12 15C13.6569 15 15 13.6569 15 12V6C15 4.34315 13.6569 3 12 3C10.3431 3 9 4.34315 9 6V12C9 13.6569 10.3431 15 12 15Z" fill="#1f2937"/>
            <path d="M19 12C19 15.866 15.866 19 12 19C8.13401 19 5 15.866 5 12M12 19V22M8 22H16" stroke="#1f2937" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
          </svg>
        </span>
        <span class="pulse-ring"></span>
      </button>
    </div>
    
    <div class="review-container" id="reviewContainer">
      <div class="review-header">
        <span>🎵</span>
        <span>Review Your Recording</span>
      </div>
      <div class="review-audio-player" id="reviewAudioPlayer">
        <!-- Audio element will be inserted here -->
      </div>
      <div class="review-buttons">
        <button class="btn btn-submit" onclick="submitRecording()">
          <span class="btn-icon">✅</span> Submit
        </button>
        <button class="btn btn-delete" onclick="deleteRecording()">
          <span class="btn-icon">❌</span> Delete
        </button>
      </div>
    </div>
  </div>
  
  <!-- Hidden input to store current task ID -->
  <?php if ($currentTask && !$allTasksCompleted): ?>
  <input type="hidden" name="task_id" id="currentTaskId" value="<?php echo htmlspecialchars($currentTask['task_id'] ?? $currentTask['id'] ?? ''); ?>">
  <?php endif; ?>
</div>

<script>
// Simple interactive mockup with recording (MediaRecorder if available),
// count-up timer, auto-scroll, pending review -> approved/rejected simulation.
const chat = document.getElementById('chat');
const micBtn = document.getElementById('micBtn');
const timerDisplay = document.getElementById('timerDisplay');
let mediaRecorder = null;
let audioChunks = [];
let recording = false;
let startTime = 0;
let timerInterval = null;
const MAX_RECORD_SEC = 6; // auto-stop after this many seconds

// Helpers: add bubbles
function addBubble({ type='user', text='', audioUrl=null, status='pending' }){
  const div = document.createElement('div');
  div.className = 'bubble ' + (type==='user' ? 'user' : (type==='instruction' ? 'instruction' : 'system'));
  if(type === 'welcome'){ div.className = 'bubble welcome'; }

  // content
  const content = document.createElement('div');
  content.innerHTML = text;
  div.appendChild(content);

  if(audioUrl){
    const audioWrap = document.createElement('div');
    audioWrap.className = 'audio-preview';
    audioWrap.innerHTML = `<audio controls src="${audioUrl}"></audio>
      <div class="meta">Uploaded • <span class="status-pending">Awaiting review...</span></div>`;
    div.appendChild(audioWrap);
  } else if(type === 'user'){
    const meta = document.createElement('div');
    meta.className = 'meta';
    meta.innerHTML = `Uploaded • <span class="status-pending">Awaiting review...</span>`;
    div.appendChild(meta);
  }

  chat.appendChild(div);
  // Scroll after animation completes
  setTimeout(() => {
    chat.scrollTop = chat.scrollHeight;
  }, 450);
  return div;
}

function formatTime(ms){
  const s = Math.floor(ms/1000);
  const mm = String(Math.floor(s/60)).padStart(2,'0');
  const ss = String(s % 60).padStart(2,'0');
  return `${mm}:${ss}`;
}

function startTimer(){
  startTime = Date.now();
  timerInterval = setInterval(() => {
    const elapsed = Date.now() - startTime;
    timerDisplay.textContent = formatTime(elapsed);
    if(elapsed >= MAX_RECORD_SEC*1000){ stopRecording(); }
  }, 200);
}

function stopTimer(){
  clearInterval(timerInterval);
  timerInterval = null;
  timerDisplay.textContent = '00:00';
}

let audioContext;
let processor;
let source;
let audioBuffer = [];

async function startRecording() {
  try {
    audioBuffer = [];
    const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
    
    // Create audio context
    const AudioContext = window.AudioContext || window.webkitAudioContext;
    audioContext = new AudioContext();
    
    // Create script processor for audio processing
    processor = audioContext.createScriptProcessor(4096, 1, 1);
    
    processor.onaudioprocess = (e) => {
      // Convert audio to mono
      const inputData = e.inputBuffer.getChannelData(0);
      audioBuffer.push(new Float32Array(inputData));
    };
    
    // Create media stream source
    source = audioContext.createMediaStreamSource(stream);
    source.connect(processor);
    processor.connect(audioContext.destination);
    
    // Start recording
    recording = true;
    micBtn.classList.add('recording');
    micBtn.setAttribute('aria-pressed', 'true');
    startTimer();
    
    // Auto-stop after max duration
    setTimeout(() => {
      if (recording) stopRecording();
    }, MAX_RECORD_SEC * 1000);
    
  } catch (err) {
    console.error('Error accessing microphone:', err);
    recording = true;
    micBtn.classList.add('recording');
    micBtn.setAttribute('aria-pressed', 'true');
    startTimer();
    console.warn('Using simulated recording due to error');
  }
}

function stopRecording() {
  // Stop recording and process audio
  if (audioContext && processor) {
    // Disconnect nodes to stop processing
    processor.disconnect();
    if (source) source.disconnect();
    
    // Process the recorded audio
    const mergedBuffer = mergeBuffers(audioBuffer, audioBuffer.length * 4096);
    const wavBlob = encodeWAV(mergedBuffer, audioContext.sampleRate);
    const url = URL.createObjectURL(wavBlob);
    
    // Handle the recorded audio
    handleRecordedAudio(url, wavBlob);
    
    // Clean up
    audioContext.close();
    audioContext = null;
    processor = null;
    source = null;
  } else {
    // Fallback to silent WAV if audio context wasn't available
    const sr = 16000;
    const duration = Math.min(MAX_RECORD_SEC, Math.max(1, Math.floor((Date.now() - startTime) / 1000)));
    const wav = makeSilentWav(duration, sr);
    const url = URL.createObjectURL(wav);
    handleRecordedAudio(url, wav);
  }
  
  // Update UI
  recording = false;
  micBtn.classList.remove('recording');
  micBtn.setAttribute('aria-pressed', 'false');
  stopTimer();
}

let currentRecordingBlob = null;
let currentRecordingUrl = null;

function handleRecordedAudio(url, blob) {
  // Store the recording
  currentRecordingBlob = blob;
  currentRecordingUrl = url;
  
  // Hide recording controls
  document.getElementById('recordingControls').style.display = 'none';
  
  // Show review container
  const reviewContainer = document.getElementById('reviewContainer');
  const reviewAudioPlayer = document.getElementById('reviewAudioPlayer');
  
  // Create audio element
  reviewAudioPlayer.innerHTML = `<audio controls src="${url}" autoplay></audio>`;
  
  // Show the review interface
  reviewContainer.classList.add('active');
}

async function submitRecording() {
  const blob = currentRecordingBlob;
  
  if (!blob) {
    console.error('No recording to submit');
    return;
  }
  
  // Get task ID from the hidden input or use a default value
  const taskInput = document.querySelector('input[name="task_id"]');
  const taskId = taskInput ? taskInput.value : '1'; // Default to task ID 1 if not found
  
  if (!taskId) {
    console.error('No task ID found');
    return; // Exit if no task ID is available
  }
  
  // Disable all buttons during submission
  const reviewContainer = document.getElementById('reviewContainer');
  const buttons = reviewContainer.querySelectorAll('button');
  const submitBtn = reviewContainer.querySelector('.btn-submit');
  
  buttons.forEach(btn => btn.disabled = true);
  
  // Show loading state
  if (submitBtn) {
    submitBtn.innerHTML = '<span class="btn-icon">⏳</span> Submitting...'
  }
  
  // Create form data and append the audio blob
  const formData = new FormData();
  
  // Ensure the blob is valid
  if (!blob || blob.size === 0) {
    console.error('Invalid or empty audio blob');
    addMessage('Recording is empty. Please try again.', 'system');
    buttons.forEach(btn => btn.disabled = false);
    if (submitBtn) submitBtn.innerHTML = '<span class="btn-icon">✅</span> Submit';
    return;
  }
  
  // Force WAV format for all recordings
  const mimeType = 'audio/wav';
  const fileExt = 'wav';
  const filename = `recording_${Date.now()}.${fileExt}`;
  
  // Convert blob to file with proper filename and type
  const audioFile = new File([blob], filename, { type: mimeType });
  
  formData.append('audio', audioFile, filename);
  formData.append('task_id', taskId);
  
  // Log the submission attempt
  console.log('Submitting recording...', {
    taskId,
    file: { 
      name: filename, 
      size: blob.size, 
      type: mimeType,
      blobType: blob.type
    }
  });
  
  try {
    // Add a small delay to ensure the UI updates
    await new Promise(resolve => setTimeout(resolve, 100));
    
    // Send the request using fetch API
    const response = await fetch('save_recording.php', {
      method: 'POST',
      body: formData,
      headers: {
        'X-Requested-With': 'XMLHttpRequest',
        'Accept': 'application/json'
      },
      credentials: 'same-origin'
    });
    
    // Log the raw response for debugging
    const responseText = await response.text();
    console.log('Raw response:', responseText);
    
    // Clean the response text to remove any HTML/notices before parsing
    const cleanResponse = responseText.replace(/^<[^>]+>/g, '').trim();
    
    // Try to parse the response as JSON
    let data;
    try {
      data = JSON.parse(cleanResponse);
    } catch (e) {
      console.error('Failed to parse JSON response:', e);
      // Try to extract error message if it's a PHP error
      const errorMatch = responseText.match(/<b>.*?<\/b>([^<]*)/);
      const errorMessage = errorMatch ? errorMatch[0].replace(/<[^>]*>/g, '') : 'Invalid server response';
      throw new Error(errorMessage);
    }
    
    if (!response.ok) {
      console.error('Server error:', data);
      throw new Error(data.message || `HTTP error! status: ${response.status}`);
    }
    
    // Show success message
    if (submitBtn) {
      submitBtn.innerHTML = '<span class="btn-icon">✅</span> Submitted!';
      submitBtn.classList.add('submitted');
    }
    
    // Add the recording to the chat
    const messageDiv = document.createElement('div');
    messageDiv.className = 'bubble user';
    
    const audioElement = document.createElement('audio');
    audioElement.controls = true;
    audioElement.src = currentRecordingUrl; // Use the blob URL we already have
    
    const metaDiv = document.createElement('div');
    metaDiv.className = 'meta';
    metaDiv.textContent = 'Just now • Awaiting review...';
    
    messageDiv.appendChild(audioElement);
    messageDiv.appendChild(metaDiv);
    
    // Add to chat and scroll to bottom
    chat.appendChild(messageDiv);
    
    // Add system response after a short delay
    setTimeout(() => {
      // First, show the approval message
      const approvalResponse = document.createElement('div');
      approvalResponse.className = 'bubble system';
      approvalResponse.innerHTML = `
        <div class="system-message">
          <span class="status-approved">✓ Approved</span>
          <div class="points-awarded">+10 points</div>
        </div>
        <div class="meta">Just now</div>
      `;
      chat.appendChild(approvalResponse);
      
      // Add a thank you message with countdown
      const taskNumber = document.querySelector('.task-counter')?.textContent || '1';
      const countdownResponse = document.createElement('div');
      countdownResponse.className = 'bubble system';
      countdownResponse.innerHTML = `
        <div class="system-message">
          <p>Thank you for completing this Task!</p>
          <div class="countdown-message">Next task available in <span class="countdown-timer">5:00</span></div>
        </div>
        <div class="meta">System</div>
      `;
      chat.appendChild(countdownResponse);
      // Scroll after animation completes
      setTimeout(() => {
        chat.scrollTop = chat.scrollHeight;
      }, 450);
      
      // Clean up
      if (currentRecordingUrl) {
        URL.revokeObjectURL(currentRecordingUrl);
      }
      currentRecordingBlob = null;
      currentRecordingUrl = null;
      
      // Hide review container and show recording controls
      document.getElementById('reviewContainer').classList.remove('active');
      document.getElementById('recordingControls').style.display = 'flex';
      
      // Hide the timer display during countdown
      const timerDisplay = document.getElementById('timerDisplay');
      if (timerDisplay) timerDisplay.style.display = 'none';
      
      // Disable record button and update countdown
      const micBtn = document.getElementById('micBtn');
      const countdownElement = countdownResponse.querySelector('.countdown-timer');
      
      if (micBtn) micBtn.disabled = true;
      
      let timeLeft = 3; // 3 seconds
      
      const updateCountdown = () => {
        if (timeLeft > 0) {
          const minutes = Math.floor(timeLeft / 60);
          const seconds = timeLeft % 60;
          const timeString = `${minutes}:${seconds.toString().padStart(2, '0')}`;
          
          // Update the countdown in the chat bubble
          if (countdownElement) {
            countdownElement.textContent = timeString;
          }
          
          // Update the mic button to show countdown
          if (micBtn) {
            micBtn.innerHTML = `<span style="font-size: 20px;">⏳</span>`;
          }
          
          timeLeft--;
          setTimeout(updateCountdown, 1000);
        } else {
          // Reload the page to show next task or congratulations message (force fresh data)
          // This ensures the Daily Task counter updates properly
          window.location.href = window.location.pathname + '?t=' + Date.now();
        }
      };
      
      updateCountdown();
    }, 1500); // Show system response after 1.5 seconds
    
    return data;
  } catch (error) {
    console.error('Submission error:', error);
    
    // Restore button state
    if (submitBtn) {
      submitBtn.innerHTML = '<span class="btn-icon">✅</span> Submit';
    }
    buttons.forEach(btn => btn.disabled = false);
    
    // Show error message in chat
    addMessage('Error: ' + (error.message || 'Failed to submit recording'), 'system');
  }
}

// Helper function to merge audio buffers
function mergeBuffers(buffers, length) {
  const result = new Float32Array(length);
  let offset = 0;
  
  for (let i = 0; i < buffers.length; i++) {
    result.set(buffers[i], offset);
    offset += buffers[i].length;
  }
  
  return result;
}

// WAV encoding functions
function encodeWAV(samples, sampleRate) {
  const numChannels = 1;
  const bytesPerSample = 2;
  const blockAlign = numChannels * bytesPerSample;
  const dataSize = samples.length * bytesPerSample;
  const buffer = new ArrayBuffer(44 + dataSize);
  const view = new DataView(buffer);

  // Helper function to write strings to the data view
  const writeString = (view, offset, string) => {
    for (let i = 0; i < string.length; i++) {
      view.setUint8(offset + i, string.charCodeAt(i));
    }
  };

  // RIFF chunk descriptor
  writeString(view, 0, 'RIFF');
  view.setUint32(4, 36 + dataSize, true); // File size - 8
  writeString(view, 8, 'WAVE');
  
  // fmt sub-chunk
  writeString(view, 12, 'fmt ');
  view.setUint32(16, 16, true);         // Subchunk1Size (16 for PCM)
  view.setUint16(20, 1, true);          // AudioFormat (1 = PCM)
  view.setUint16(22, numChannels, true);
  view.setUint32(24, sampleRate, true);
  view.setUint32(28, sampleRate * blockAlign, true); // ByteRate
  view.setUint16(32, blockAlign, true);  // BlockAlign
  view.setUint16(34, 16, true);         // BitsPerSample (16-bit)
  
  // data sub-chunk
  writeString(view, 36, 'data');
  view.setUint32(40, dataSize, true);   // Subchunk2Size

  // Write PCM samples
  let offset = 44;
  for (let i = 0; i < samples.length; i++, offset += 2) {
    const s = Math.max(-1, Math.min(1, samples[i])); // Clamp to [-1, 1]
    view.setInt16(offset, s < 0 ? s * 0x8000 : s * 0x7FFF, true);
  }

  // Create blob with explicit MIME type that matches the server's expectations
  return new Blob([view], { type: 'audio/wav' });
}

// Fallback function to create a silent WAV file
function makeSilentWav(duration, sampleRate) {
  const numSamples = duration * sampleRate;
  const buffer = new ArrayBuffer(44 + numSamples * 2);
  const view = new DataView(buffer);
  
  // RIFF chunk descriptor
  writeString(view, 0, 'RIFF');
  view.setUint32(4, 36 + numSamples * 2, true);
  writeString(view, 8, 'WAVE');
  
  // fmt sub-chunk
  writeString(view, 12, 'fmt ');
  view.setUint32(16, 16, true);
  view.setUint16(20, 1, true);
  view.setUint16(22, 1, true);
  view.setUint32(24, sampleRate, true);
  view.setUint32(28, sampleRate * 2, true);
  view.setUint16(32, 2, true);
  view.setUint16(34, 16, true);
  
  // data sub-chunk
  writeString(view, 36, 'data');
  view.setUint32(40, numSamples * 2, true);
  
  // Write silent samples
  for (let i = 0; i < numSamples; i++) {
    view.setInt16(44 + (i * 2), 0, true);
  }
  
  return new Blob([view], { type: 'audio/wav' });
}

function writeString(view, offset, string) {
  for (let i = 0; i < string.length; i++) {
    view.setUint8(offset + i, string.charCodeAt(i));
  }
}

function deleteRecording() {
  // Revoke the blob URL to free memory
  if (currentRecordingUrl) {
    URL.revokeObjectURL(currentRecordingUrl);
  }
  
  // Clear stored recording
  currentRecordingBlob = null;
  currentRecordingUrl = null;
  
  // Hide review container and show recording controls
  document.getElementById('reviewContainer').classList.remove('active');
  document.getElementById('recordingControls').style.display = 'flex';
  
  // Add a system message
  addMessage('Recording deleted. Please try again.', 'system');
}

function loadNextTask() {
  // Reload the page to get the next task
  window.location.reload();
}

// helper: create tiny silent WAV blob (for simulated recording)
function makeSilentWav(seconds=1, sampleRate=16000){
  const numSamples = seconds * sampleRate;
  const buffer = new ArrayBuffer(44 + numSamples * 2);
  const view = new DataView(buffer);
  writeString(view, 0, 'RIFF'); view.setUint32(4, 36 + numSamples * 2, true);
  writeString(view, 8, 'WAVE');
  writeString(view, 12, 'fmt '); view.setUint32(16, 16, true);
  view.setUint16(20, 1, true); view.setUint16(22, 1, true);
  view.setUint32(24, sampleRate, true); view.setUint32(28, sampleRate*2, true);
  view.setUint16(32, 2, true); view.setUint16(34, 16, true);
  writeString(view, 36, 'data'); view.setUint32(40, numSamples*2, true);
  let offset = 44;
  for(let i=0;i<numSamples;i++){ view.setInt16(offset,0,true); offset+=2; }
  return new Blob([view], { type:'audio/wav' });
}

function writeString(view, offset, string){ 
  for(let i=0;i<string.length;i++){ 
    view.setUint8(offset+i,string.charCodeAt(i)); 
  } 
}

// Event Listeners
micBtn.addEventListener('click', async () => {
  if(!recording){ 
    await startRecording(); 
  } else { 
    stopRecording(); 
  }
});

// Auto-scroll observer with proper timing for animations
function smoothScrollToBottom() {
  requestAnimationFrame(() => {
    chat.scrollTop = chat.scrollHeight;
  });
}

const observer = new MutationObserver(() => {
  smoothScrollToBottom();
  // Double check after animation completes
  setTimeout(smoothScrollToBottom, 450);
});
observer.observe(chat, { childList: true, subtree: true });

// Initial scroll
setTimeout(() => { chat.scrollTop = chat.scrollHeight; }, 200);

// Confetti animation for celebration screen
function createConfetti() {
  const confettiContainer = document.getElementById('confetti');
  if (!confettiContainer) return;
  
  const colors = ['#FFC929', '#FFD33D', '#ffffff', '#FFB800', '#fbbf24'];
  const confettiCount = 80;
  
  for (let i = 0; i < confettiCount; i++) {
    setTimeout(() => {
      const confetti = document.createElement('div');
      confetti.className = 'confetti-piece';
      
      // Random properties
      const startX = Math.random() * 100;
      const endX = (Math.random() - 0.5) * 200;
      const color = colors[Math.floor(Math.random() * colors.length)];
      const size = Math.random() * 8 + 6;
      const duration = Math.random() * 2 + 2;
      const delay = Math.random() * 0.5;
      
      confetti.style.left = startX + '%';
      confetti.style.width = size + 'px';
      confetti.style.height = size + 'px';
      confetti.style.background = color;
      confetti.style.setProperty('--confetti-x', endX + 'px');
      confetti.style.animationDuration = duration + 's';
      confetti.style.animationDelay = delay + 's';
      
      // Random shape
      if (Math.random() > 0.5) {
        confetti.style.borderRadius = '50%';
      }
      
      confettiContainer.appendChild(confetti);
      
      // Remove after animation
      setTimeout(() => {
        confetti.remove();
      }, (duration + delay) * 1000 + 100);
    }, i * 30);
  }
  
  // Repeat confetti every 4 seconds
  setTimeout(createConfetti, 4000);
}

// Start confetti if celebration container exists
if (document.getElementById('celebrationContainer')) {
  setTimeout(createConfetti, 500);
}
</script>
</body>
</html>
